// 定数
final float MAX_ANGLE  = 75; // 関節の最大回転角度（あくまでGUI用であって，実際の角度はArduino側で変更する必要がある） [deg]
final float EASY_ANGLE = 75; // Easyモードで横移動する際の関節角度（あくまでGUI用であって，実際の角度はArduino側で変更する必要がある） [deg]
final float STEP_ANGLE = 15; // 一定角度単位で関節を回転する際の相対関節角度（同上）

// ゲームパッド＆キーボード操作と本体の状態に基づいてロボットへの制御コマンドを送信する
// 戻り値の型にRobotControlが指定されている（return this）関数は，簡易自動コマンド中で利用可能（マルチステートメント対応）
class RobotControl {
  char current_command_flag;  // 最後に出力したコマンド

  // コンストラクター
  RobotControl() {
  }
  // コマンドフラグをリセット
  void reset() {
    command_flag = 0b11110000;  // 移動用モーターは停止，関節部は未定義状態をデフォルトとする．
  }
  // フラグを送信（簡易自動コマンド内で用いる）
  RobotControl doing() {
    serial_write(command_flag);
    return this;
  }
  RobotControl wait(int ms) {
    SM.set_message(ms + "[ms]待ちます");
    delay(ms);
    return this;
  }
  // 全モーター停止
  RobotControl all_motor_stop() {
    command_flag = 0b11111111;
    serial_write(command_flag);
    SM.set_message("全モーターを停止します");
    return this;
  }
  // 全駆動モーター停止
  RobotControl robot_stop() {
    command_flag |= 0b11110000; 
    serial_write(command_flag);
//  SM.set_message("ロボット停止");
    return this;
  }
  // 関節モーター停止
  RobotControl joint_stop() {
    command_flag &= 0b11110000;
    command_flag |= 0b00000011;
//  serial_write(command_flag);  // 副作用が出やすいのでコマンド送出を都度行うのは休止する Ver.1.1.6
//  SM.set_message("関節モーター停止");
    return this;
  }
  // 前方モーター前進
  RobotControl robot_A_fwd() {
    command_flag &= 0b00111111;
    command_flag |= 0b10000000;
//  SM.set_message("前方モーター前進");
    return this;
  }
  // 前方モーター後進
  RobotControl robot_A_bwd() {
    command_flag &= 0b00111111;
    command_flag |= 0b01000000;
//  SM.set_message("前方モーター後進");
    return this;
  }
  // 後方モーター前進
  RobotControl robot_B_fwd() {
    command_flag &= 0b11001111;
    command_flag |= 0b00100000;
//  SM.set_message("後方モーター前進");
    return this;
  }
  // 後方モーター後進
  RobotControl robot_B_bwd() {
    command_flag &= 0b11001111;
    command_flag |= 0b00010000;
//  SM.set_message("後方モーター後進");
    return this;
  }
  // ロボット前進
  RobotControl robot_fwd() {
    robot_A_fwd();
    robot_B_fwd();
    SM.set_message("ロボット前進");
    return this;
  }
  // ロボット後進
  RobotControl robot_bwd() {
    robot_A_bwd();
    robot_B_bwd();
    SM.set_message("ロボット後進");
    return this;
  }
  // 関節角度をゼロ度に戻す
  RobotControl joint_angle_zero() {
    command_flag &= 0b11110000;
    command_flag |= 0b00001100;
    rs.JT_Angle = 0.0;
    SM.set_message("ロボットの姿勢を真直ぐにしています");
    return this;
  }
  // 関節を相対的にSTEP_ANGLE度回転（CCW）
  RobotControl joint_ccw_15deg() {
    command_flag &= 0b11110000;
    command_flag |= 0b00000100;
    if ((rs.JT_Angle += STEP_ANGLE) >  MAX_ANGLE) rs.JT_Angle =  MAX_ANGLE;
    SM.set_message("CCW+" + STEP_ANGLE + "度回転");             
    return this;
  }
  // 関節を相対的にSTEP_ANGLE度回転（CW）
  RobotControl joint_cw_15deg() {
    command_flag &= 0b11110000;
    command_flag |= 0b00001000;
    if ((rs.JT_Angle -= STEP_ANGLE) < -MAX_ANGLE) rs.JT_Angle = -MAX_ANGLE;
    SM.set_message("CW-" + STEP_ANGLE + "度回転");
    return this;
  }
  // 関節をEASY_ANGLE度に回転（CCW）  // Easyモード時の「くの字」の角度
  RobotControl joint_right_EASY() {
    command_flag &= 0b11110000;
    command_flag |= 0b00000101;
    rs.JT_Angle =  EASY_ANGLE;
    SM.set_message(EASY_ANGLE + "度まで回転(CCW)");             
    return this;
  }
  // 関節をEASY_ANGLE度まで回転（CW）
  RobotControl joint_left_EASY() {
    command_flag &= 0b11110000;
    command_flag |= 0b00001010;
    rs.JT_Angle = -EASY_ANGLE;
    SM.set_message("－" + EASY_ANGLE + "度まで回転(CW)");             
    return this;
  }
  // 関節をCCWに回転（連続）
  RobotControl joint_ccw() {
    command_flag &= 0b11110000;
    command_flag |= 0b00000001;
    if (++rs.JT_Angle >  MAX_ANGLE) rs.JT_Angle =  MAX_ANGLE;
    SM.set_message("CCW回転中");
    return this;
  }
  // 関節をCWに回転（連続）
  RobotControl joint_cw() {
    command_flag &= 0b11110000;
    command_flag |= 0b00000010;
    if (--rs.JT_Angle < -MAX_ANGLE) rs.JT_Angle = -MAX_ANGLE;
    SM.set_message("CW回転中");
    return this;
  }
  // 関節角度を初期化
  RobotControl joint_reset() {
    command_flag &= 0b11110000;
    command_flag |= 0b00001001;
    rs.JT_Angle = 0.0;
    SM.set_message("関節角度初期化");
    return this;
  }
  // ゲームパッドやキーボード操作に基づいて，ロボットへコマンドを送信する
  void robotController() {
    int     read = serial_read();
    
    reset();  // command_flagをリセット
    if (mc.hat == 0) {
      // 各種オプションのキーは，ハットキーが押されていない時にのみ判定される
      if (mc.button_X) { // モード切替
        if (! advance_mode) {
          advance_mode = true;
          SM.set_message("Advance modeです", true, false);
        } else {
          advance_mode = false;
          SM.set_message("Easy modeです", true, false);
        }            
      } else if (mc.button_A) {  // 簡易自動コマンドを実行する
        if (mc.autoCommand > 0) {
          SM.set_message("簡易自動コマンド(" + mc.autoCommand + ")を実行します", true, false);
          mc.run_autoCommand();
          rc.all_motor_stop();    // 安全の為に全モーターを停止する．
          SM.set_message("簡易自動コマンド終了", true, false);
        }
      } else if (ksup.home) rc.joint_reset();  // ポテンショメーター初期化
    } else if (mc.hat == 2) robot_fwd(); // HAT：上（前進）
    else   if (mc.hat == 6) robot_bwd(); // HAT：下（後進）
    else   if (mc.hat == 4){             // HAT：右
      if (advance_mode) {
        joint_stop();
        if (rs.JT_Angle >= 0) {  // Easyと同様
          robot_A_bwd(); robot_B_fwd();
          SM.set_message("右に並進移動します", true, false);
        } else {                 // 逆関節なのでA/B反転
          robot_A_fwd(); robot_B_bwd();
          SM.set_message("右に並進移動します（反転）", true, false);
        }
      } else {
        robot_A_bwd(); robot_B_fwd();
        joint_right_EASY();
        SM.set_message("右に並進移動します", true, false);
      }
    } else if (mc.hat == 8) {            // HAT：左
      if (advance_mode){
        joint_stop();
        if (rs.JT_Angle <= 0) {  // Easyと同様
          robot_A_bwd(); robot_B_fwd();
          SM.set_message("左に並進移動します", true, false);
        } else {                 // 逆関節なのでA/B反転
          robot_A_fwd(); robot_B_bwd();
          SM.set_message("左に並進移動します（反転）", true, false);
        }
      } else {
        robot_A_bwd(); robot_B_fwd();
        joint_left_EASY();
        SM.set_message("左に並進移動します", true, false);
      }
    }
    // 関節部の制御（駆動モーターと並行）
    if (mc.button_LB)      joint_ccw_15deg();  // 相対的に１５度回転（CCW）
    else if (mc.button_RB) joint_cw_15deg();   // 相対的に１５度回転（CW）
    else if (mc.button_LT) joint_ccw();        // 関節を連続回転（CCW）
    else if (mc.button_RT) joint_cw();         // 関節を連続回転（CW）
    else if (mc.button_Y)  joint_angle_zero(); // 直線姿勢変形
    // 特例：LT(1), RT(3)が離された時に関節部モーターを停止する
    if ((mc.button_RT0 && ! mc.button_RT) || ksdown.tenkey[1]) joint_stop();
    if ((mc.button_LT0 && ! mc.button_LT) || ksdown.tenkey[3]) joint_stop();

    // 全モーター停止（最優先）
    if (mc.button_B) all_motor_stop();

    // コマンド送信
    serial_write(command_flag);
    
    // キーボードの立ち上がり／立下り検出をリセット
    ksup.reset();
    ksdown.reset();
  }
}

RobotControl rc = new RobotControl();
